/*
*  Arnold emulator (c) Copyright, Kevin Thacker 1995-2015
*
*  This file is part of the Arnold emulator source code distribution.
*
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

/* Bryce mentioned that the LA7800 has an internal horizontal and vertical oscillator */

/* Gerald:

"The horizontal retrace is done by a VCO (voltage controlled oscillator) that locks on the incoming horizontal sync.
When in locked, you can consider that the retrace start as soon as the sync pulse is received. The retrace time is a constant of the monitor/TV. It is not influenced by the length of the sync pulse.
When out of sync, the retrace may start later/earlier according to the reaction time of the VCO. The retrace is done on the VCO timing, not the incoming vsync timing."

Retrace time is fixed.

No equalizing pulses generated by CPC.

Gamma is the non-linearity of the electron beam.

Fixed range. This is function of the horizontal VCO min/max frequency and free running vertical frequency. I cannot remember if there is a hold of timer for the VSYNC/HSYNC, preventing too fast SYNC.

The frequency of the HSYNC VCO is adjusted to match the average of incoming HSYNC.

Yes, that's the circuit that will adjust the VCO according to the timing differences.
*/

/* 
NOTES

modern
television
practice
priniciples technology and servicing
second edition
r.r. gulati

brightness varies beam intensity
contrast is the gain

r,g,b gun

video amplifier = y signal

4/3 aspect ratio

light persistence of about 1/16 of a second

linear rise of current in the horizontal line deflection coils uniform left/right.
at peak of it's rise, sawtooth wave reverses direction and
decreases rapidly to initial value. this is the flyback.

trace part of the sawtooth wave

horizontal and vertical retrace are blanked.

15625 horizontasl in 625
50hz frequency

scanning periods

64us nominal
active 52
12us blanking

20ms retrace

20lines to reach top

front porch 1.5us, cpc gives 2
back porch
line sync 4.7us retrace
back porch 5.8

5ms persistence.

contrast = amplitude
brightness = offset


afc has a slow varying voltage.
zero if equal to 15625

sawtooth and sync simultaneous: freq correct.
oscillator more than sync
oscillator less than sync

more: sync pulses arrive when sawtooth already in positive half cycle.
biases d2 and reverses d1

less: sync pulses happen in negative half cycle.


locks horizontal oscillator at synchronizing frequency.

LA7800:
Horizontal AFC
Horizontal oscillation
Vertical oscillation

horizontal output pulse width: 21.5 -> 26.5
*/

/* 
BRIGHTNESS:
- Within all the colour monitors, R,G,B is summed to generate a "brightness" reference.
- In the mono monitors luminance is used to generate a "brightness" reference.
- The brightness is controlled through a variable resistor under user control. Often marked VR401. It is the brightness dial on the side of the CTM644, on the front of the CM14, MM14, GT64 and GT65.
- The lower bound is derived from the brightness through a fixed VR402 variable resistor that can be adjusted on the PCB.
- I can't seem to find the exact answer. But it appears to have a logarithmic scale. CTM644 calls it a "volume" variable resistor and volume perception is known to be logarithmic.
Also brightness is perceived as logarithmic too. Therefore it is assumed it *is* logarithmic */

/* GAMMA:

"A cathode-ray tube (CRT) is inherently nonlinear: The intensity of light reproduced at the screen of a CRT monitor is a nonlinear function of its voltage input. " from,"Gamma" topic in
"A Technical Introduction to Digital Video" by Charles Poynton, published in 1996 by John Wiley & Sons.
*/

/* CONTRAST:
- Colour monitors don't have contrast control, there is nothing user adjustable.
- Mono monitors have contrast control. They appear to take the luminance input and adjust it using a logarithmic scale.
Contrast is done BEFORE brightness */

/* COLOUR: B appears to have a different value compared to red and green, doesn't that mean it is slightly brighter? 6.7 compared to 6.5 in CM14 */

/* TUBE:
- R, G,B are present at the tube as 3 seperate inputs for the 3 seperate guns.
- Another input, is controlled by the brightness, thereby controlling the brightness for ALL 3 colours at the same time. This is also probably the heater which controls
the emission of the electrons. The brightness is therefore a scale factor or a multiplied number. This means maximum brightness is where R,G,B are unaltered.
The brightness is probably directly related to the gamma?
*/

#include "cpc.h"
#include "render.h"
#include "garray.h"
#include "monitor.h"
#include "aleste.h"
#include <math.h>
float BlackLevel = 0.01f;
float BlackLevelR = 0.01f;
float BlackLevelG = 0.01f;
float BlackLevelB = 0.01f;

//float BrightnessLower = 0.0f;	// from PCB the "sub-brightness"; NOT USED
float MonitorContrast = 1.0f;
//float  MonitorBrightness = 1.0f;	// from the dial (between 0 and 1) 0 being min brightness, 1 being max brightness

float GammaSource = 2.2f;	
float GammaDest = 2.5f;	//2.5; // NTSC, but PAL is 2.8

MONITOR_INTERNAL_STATE Monitor_State;

void Monitor_SetContrast(int Amount)
{
	if (Amount < 0)
		Amount = 0;
	if (Amount > 100)
		Amount = 100;
	BlackLevel = Amount / 100.0f;
	BlackLevelR = Amount / 100.0f;
	BlackLevelG = Amount / 100.0f;
	BlackLevelB = Amount / 100.0f;
}

void Monitor_SetBrightness(int Amount)
{
	if (Amount < 0)
		Amount = 0;
	if (Amount > 100)
		Amount = 100;
	MonitorContrast = Amount / 100.0f;
}

int Monitor_GetBrightness(void)
{
	return (int)floorf(MonitorContrast * 100);

}

int Monitor_GetContrast(void)
{
	return (int)floorf(BlackLevel* 100);

}

/* ---------------------------------------------------------------------------------------------------------- */

BOOL Monitor_GetDrawSync(void)
{
	return Monitor_State.bDrawSyncs;
}

/* ---------------------------------------------------------------------------------------------------------- */

void Monitor_EnableDrawSync(BOOL bEnableDrawSync)
{
	Monitor_State.bDrawSyncs = bEnableDrawSync;
}

int Monitor_AdjustLuminanceForDisplay(int Luminance)
{
#if 1
	return Luminance;
#else

	float GammaCorrection = 1.0f/2.5f;

	float BrightnessInput = (float)Luminance/255.0f;
	//printf("Brightness Input: %f\n", BrightnessInput);
	
	// apply contrast
	BrightnessInput *= MonitorContrast;
	//printf("Brightness post contrast: %f\n", BrightnessInput);
	// apply black level	
	BrightnessInput += BlackLevel;
	//printf("Brightness post black level: %f\n", BrightnessInput);

	// now we do the gamma correction
	BrightnessInput = powf(BrightnessInput, GammaCorrection);
//	//printf("Brightness Output post gamma: %f\n", BrightnessOutput);
	
	/* clamp for display */
	if (BrightnessInput<0.0f)
	{
		BrightnessInput = 0.0f;
	}
	
	if (BrightnessInput>1.0f)
	{
		BrightnessInput = 1.0f;
	}
	
	Luminance = (int)(BrightnessInput*255.0f);
	
//	//printf("Monitor Luminance: %d\n", Luminance);
	return Luminance;
#endif
}

/* both CTM644 and CM14 are calibrated based on green */
/* they have different values */
void Monitor_AdjustRGBForDisplay(RGBCOLOUR *Colour)
{
#if 0
	float GammaCorrection = 1.0f/2.5f;
//	float GammaCorrection = (GammaDest/GammaSource);

	////printf("RI: %02x GI: %02x BI: %02\n", Colour->u.element.Red, Colour->u.element.Green, Colour->u.element.Blue);

	float R,G,B;
	/* this is effectively white balancing */
	R = Colour->u.element.Red/255.0f;
	G = Colour->u.element.Green/255.0f;
	B = Colour->u.element.Blue/255.0f;

	/* brightness from summing r,g,b */
	float BrightnessInput = (R+G+B)/3.0f;
	
	////printf("Brightness input: %f\n", BrightnessInput);
	
	BrightnessInput *= MonitorContrast;
	
	/* gamma correction per colour or on the brightness? */

	// now we do the gamma correction
	BrightnessInput = powf(BrightnessInput, GammaCorrection);

	R = (R*BrightnessInput) + BlackLevelR;
	G = (G*BrightnessInput) + BlackLevelG;
	B = (B*BrightnessInput) + BlackLevelB;
	
	// intensity = ( voltage + black_level)^2.5
	// signal = pow((double)intensity,(double)1.0/gamma)
	
	// clamp for display
	if (R<0.0f)
	{
		R = 0.0f;
	}
	if (G<0.0f)
	{
		G = 0.0f;
	}
	if (B<0.0f)
	{
		B = 0.0f;
	}
	
	if (R>1.0f)
	{
		R = 1.0f;
	}
	if (G>1.0f)
	{
		G = 1.0f;
	}
	if (B>1.0f)
	{
		B = 1.0f;
	}
	
	Colour->u.element.Red=(int)floorf( 255.0f*R);
	Colour->u.element.Green=(int)floorf( 255.0f*G);
	Colour->u.element.Blue=(int)floorf( 255.0f*B);
	
	////printf("RO: %02x GO: %02x BO: %02x\n", Colour->u.element.Red, Colour->u.element.Green, Colour->u.element.Blue);
#endif
}


/* ---------------------------------------------------------------------------------------------------------- */

/* get current state of vsync signal into monitor */
BOOL Monitor_GetVsyncState(void)
{
	return ((Monitor_State.MonitorSyncInputs & VSYNC_INPUT)!=0);
}

/* ---------------------------------------------------------------------------------------------------------- */

/* get current state of hsync signal into monitor */
BOOL Monitor_GetHsyncState(void)
{
	return ((Monitor_State.MonitorSyncInputs & HSYNC_INPUT)!=0);
}

/* ---------------------------------------------------------------------------------------------------------- */

BOOL Monitor_DrawSync(void)
{
	/* force no sync drawing */
	if (!Monitor_State.bDrawSyncs)
	{
		return FALSE;
	}
		
	/* hsync or vsync inputs are active to monitor */
	return ((Monitor_State.MonitorSyncInputs & (VSYNC_INPUT|HSYNC_INPUT))!=0);
}

#if 0
void Monitor_DoInternalUpdate(void)
{
	
	/* has vsync state changed and become active? */
	if (Monitor_State.MonitorSyncInputs & VSYNC_INPUT)
	{
		/* TODO: This doesn't actually relate to the HSYNC end */

		/* monitor accepting vsync and beginning trace */
		Monitor_State.MonitorVTraceActive = TRUE;
		/* the number of lines to trace for */
		Monitor_State.MonitorVTraceCount = MONITOR_VTRACE_COUNT;
	}

/* if monitor v trace is active we count it */
    if (Monitor_State.MonitorVTraceActive)
    {
	Monitor_State.MonitorVTraceCount--;
	Monitor_State.MonitorScanLineCount = 0;
	Monitor_State.MonitorHorizontalCount = 0;
	    
        if (Monitor_State.MonitorVTraceCount==0)
        {
                Monitor_State.MonitorVTraceActive = FALSE;
        }
    }
	

	/* monitor vsync */

	/* if we want to perform a monitor vsync, do this */
	if (Monitor_State.MonitorVTraceActive)
	{
		if (Monitor_State.LinesAfterVsync>=312)
		{
			Monitor_State.LinesAfterVsync = 0;

			/* for monitor sync */
			Monitor_State.MonitorHorizontalCount = 0;

			Monitor_State.MonitorScanLineCount = 0;

		}
	}
	
	/* seen end of hsync */
	if (Monitor_State.CharsAfterHsync>=64)
	{
		Monitor_State.
		/* reset */
		Monitor_State.CharsAfterHsync=0;

		/* reset count */
		Monitor_State.MonitorHorizontalCount = 0;

		/* increment line */
		Monitor_State.MonitorScanLineCount++;

		if (Monitor_State.MonitorScanLineCount>=(39*8))
		{
			Monitor_State.MonitorScanLineCount = 0;
		}
	}
}
#endif


static BOOL bMonitorUpdate = FALSE;
BOOL MonitorNeedupdate(void)
{
	if (bMonitorUpdate)
	{
		bMonitorUpdate = FALSE;
		return TRUE;
	}
	return FALSE;
}

/* ---------------------------------------------------------------------------------------------------------- */

/* called when vsync input begins */
void Monitor_DoVsyncStart(void)
{
	/* indicate vsync input active */
	if ((Monitor_State.MonitorSyncInputs & VSYNC_INPUT) == 0)
	{
		Monitor_State.MonitorSyncInputs |= VSYNC_INPUT;

		/* hard sync */
		Monitor_State.MonitorVCount = MONITOR_HEIGHT_LINES;
		Monitor_State.MonitorVTraceActive = TRUE;
		Monitor_State.MonitorVTraceCount = MONITOR_VTRACE_COUNT;

		bMonitorUpdate = TRUE;
	}
}


/* ---------------------------------------------------------------------------------------------------------- */

/* called when vsync input ends */
void Monitor_DoVsyncEnd(void)
{
	if ((Monitor_State.MonitorSyncInputs & VSYNC_INPUT) != 0)
	{
		/* indicate vsync input inactive */
		Monitor_State.MonitorSyncInputs &= ~VSYNC_INPUT;
	}
}

/* ---------------------------------------------------------------------------------------------------------- */

/* called when hsync inputs starts */
void Monitor_DoHsyncStart(void)
{
	/* only start if not already started */
	if ((Monitor_State.MonitorSyncInputs & HSYNC_INPUT) == 0)
	{
		/* indicate hsync is active */
		Monitor_State.MonitorSyncInputs |= HSYNC_INPUT;

		/* hard sync */
		Monitor_State.MonitorHCount = MONITOR_WIDTH_CHARS;
		Monitor_State.MonitorHTraceActive = TRUE;
		Monitor_State.MonitorHTraceCount = MONITOR_HTRACE_COUNT;
	}

	/* to sync, we need to move the htrace start backwards and forwards */

	/* htrace could be speeded up or slowed down which would then allow the input to sync */
	/* we could also start the sequence faster or slower.. */
}

/* ---------------------------------------------------------------------------------------------------------- */

/* called when hsync input ends */
void Monitor_DoHsyncEnd(void)
{
	/* only end if not already ended */
	if ((Monitor_State.MonitorSyncInputs & HSYNC_INPUT) != 0)
	{
		/* indicate hsync is inactive */
		Monitor_State.MonitorSyncInputs &= ~HSYNC_INPUT;


	}

}



/* ---------------------------------------------------------------------------------------------------------- */

void Monitor_Reset(void)
{
	Monitor_State.bDrawSyncs = TRUE;

//	Monitor_State.CharsAfterHsync = 0;
//	Monitor_State.LinesAfterVsync = 0;
	Monitor_State.MonitorVCount = 0;
	Monitor_State.MonitorHCount = 0;
	Monitor_State.MonitorSyncInputs = 0;
	Monitor_State.MonitorHorizontalCount = 0;
	Monitor_State.MonitorScanLineCount = 0;
	Monitor_State.MonitorVTraceActive = FALSE;
	Monitor_State.MonitorVTraceCount = 0;
	Monitor_State.MonitorHTraceActive = FALSE;
	Monitor_State.MonitorHTraceCount = 0;
	Monitor_State.LinesAfterVTrace = 0;
	Monitor_State.CharsAfterHTrace = 0;
}

void Monitor_Init(void)
{
	Monitor_Reset();
}
/* ---------------------------------------------------------------------------------------------------------- */

/* called once for each crtc cycle */ 
void	Monitor_Cycle()
{
	/* horizontal trace and vertical trace operate at the same time */
	/* TODO: Change vertical trace to be cycles and not lines */
	/* monitor does go black if hsync goes missing */
	if (Monitor_State.MonitorVTraceActive)
	{
		Monitor_State.MonitorVTraceCount--;
		if (Monitor_State.MonitorVTraceCount == 0)
		{
			Monitor_State.MonitorVCount = 0;
			Monitor_State.MonitorVTraceActive = FALSE;
			Monitor_State.MonitorScanLineCount = 0;
			//printf("Vtrace inactive\n");
		}
	}

	{
		if (Monitor_State.MonitorHTraceActive)
		{
			Monitor_State.MonitorHTraceCount--;
			if (Monitor_State.MonitorHTraceCount == 0)
			{
			//	printf("Monitor HTrace inactive \n");
				Monitor_State.MonitorHTraceActive = FALSE;

				/* go to next line */
				if (Monitor_State.MonitorVCount == MONITOR_HEIGHT_LINES)
				{
					/* draw position */
					Monitor_State.LinesAfterVTrace = 0;
					Monitor_State.MonitorVTraceActive = TRUE;
					Monitor_State.MonitorVTraceCount = MONITOR_VTRACE_COUNT;

				//	printf("Vtrace active\n");
				}

				Monitor_State.MonitorVCount++;
				Monitor_State.LinesAfterVTrace++;
				Monitor_State.MonitorHCount = 0;
				Monitor_State.MonitorScanLineCount++;
			}
		}
		else
		{
			Monitor_State.MonitorHorizontalCount++;
			
			Monitor_State.MonitorHCount++;
			if (Monitor_State.MonitorHCount == MONITOR_WIDTH_CHARS)
			{
				Monitor_State.MonitorHTraceActive = TRUE;
				Monitor_State.MonitorHTraceCount = MONITOR_HTRACE_COUNT;
				Monitor_State.CharsAfterHTrace = 0;
				Monitor_State.MonitorHorizontalCount = 0;
				//printf("Monitor HTrace active\n");
			}

		}
	}
	
	/* when tracing the beam is blanked on a real television.
	We park the draw position */
	if (Monitor_State.MonitorVTraceActive)
	{
		Monitor_State.MonitorScanLineCount = 311;
	}
	if (Monitor_State.MonitorHTraceActive)
	{
		Monitor_State.MonitorHorizontalCount = 63;
	}
}

/* ---------------------------------------------------------------------------------------------------------- */









/* ---------------------------------------------------------------------------------------------------------- */
/* ---------------------------------------------------------------------------------------------------------- */
/* ---------------------------------------------------------------------------------------------------------- */
/* ---------------------------------------------------------------------------------------------------------- */

extern void CRTC_RenderSync_TrueColour(void);
extern void CRTC_RenderBorder_TrueColour(void);

/* rendering function */
static void (*pCRTC_RenderFunction)(void) =  CRTC_RenderSync_TrueColour;

void CRTC_SetRenderFunction2(void (*pRenderFunction)(void))
{
	pCRTC_RenderFunction = pRenderFunction;
}

BOOL Monitor_RendererReady = FALSE;

void Monitor_SetRendererReady(BOOL bState)
{
	Monitor_RendererReady = bState;
}

/* called when crtc ticks and graphics need to update */
void	Graphics_Update(void)
{
	Monitor_Cycle();

	if (Monitor_RendererReady)
	{
		pCRTC_RenderFunction();
	}
}

























/* CLEAN ME UP */
#if 0

/*

17 moves

136 is too small

144-160

18-20 is ok. 21 moves

168 is too big

range appears to be about 24 lines

little bit of black, but also something a bit brighter

monitor has a specific range it can accept.


monitor wants a range around 312 lines

say 312 is ideal, min is 300, max is 324

625 lines - 312.5 lines
576 lines of display, 46 lines for retrace

288 scanlines of display
46 retrace = 23 lines


*/

int MinLineVert = 288;
int MaxLineVert = 312;

#include "cpc.h"
#include "render.h"
#include "garray.h"
/*#define LESS_MULTS */
#include "monitor.h"
#include "aleste.h"

#define MONITOR_IN_VSYNC	0x0001
#define MONITOR_VSYNC_SYNC	0x0002
#define MONITOR_IN_HSYNC	0x0004
#define MONITOR_DRAW_HSYNC  0x0008
#define MONITOR_DRAW_VSYNC  0X0010

/* number of monitor lines for which vsync is drawn */
#define MONITOR_VSYNC_COUNT	(24+1)


/*unsigned long GraphicsLong;
unsigned long CRTC_RenderType;
*/
/* render graphics */
/*void (*pCRTC_RenderGraphicsFunction)(void);

void (*pCRTC_RenderSync)(void);
void (*pRender_RenderBorder)(void);

void (*pRender_GetGraphicsDataCPC)(void);
void (*pRender_GetGraphicsDataPLUS)(void); */
static	BOOL DontRender;

static 	BOOL RenderSync = TRUE;

extern void CRTC_RenderSync_TrueColour(void);

/* rendering function */
static void (*pCRTC_RenderFunction)(void) =  CRTC_RenderSync_TrueColour;

void CRTC_SetRenderFunction2(void (*pRenderFunction)(void))
{
	pCRTC_RenderFunction = pRenderFunction;
}

MONITOR_INTERNAL_STATE Monitor_State;

void Montor_SetRenderSync(BOOL bState)
{
	RenderSync = bState;
}

BOOL Monitor_GetRenderSync()
{
	return RenderSync;
}

void Monitor_UpdateGraphicsFunc()
{
	Monitor_State.bDrawSync = FALSE;
	if (RenderSync)
	{
		if ((Monitor_State.MonitorFlags & MONITOR_DRAW_HSYNC)!=0)
		{
			Monitor_State.bDrawSync = TRUE;

		}

		if ((Monitor_State.MonitorFlags & MONITOR_DRAW_VSYNC)!=0)
		{
			Monitor_State.bDrawSync = TRUE;

		}
	}

	Computer_UpdateGraphicsFunction();
}


/*#define SIMPLE_MONITOR_EMULATION */

#if 0
void	CRTC_MonitorReset(void);
void	CRTC_Monitor_NewFrame(void);
static void	CRTC_NextRenderStage(unsigned long Flags);
static void CRTC_FirstRenderStage(void);
static void     Render_GetGraphicsDataCPC_TrueColour(void);
static void     Render_GetGraphicsDataPLUS_TrueColour(void);
static void     Render_RenderBorder_TrueColour(void);
static void		CRTC_RenderSync_TrueColour(void);

static void     Render_GetGraphicsDataCPC_Paletted(void);
static void     Render_GetGraphicsDataPLUS_Paletted(void);
static void     Render_RenderBorder_Paletted(void);
static void		CRTC_RenderSync_Paletted(void);


void    Render_RenderBorder_Aleste_TrueColour(void);
        /* clear display so that un-rendered bits are black - necessary for Nega Part of "The
      Demo" currently. Some programs do not like it when the CRTC is changed, change CRTC
    at Vsync or VTot?*/
  Render_ClearDisplay();
#endif

void Monitor_Reset()
{
	Monitor_State.bDispEnable = FALSE;
	Monitor_State.MonitorFlags = 0;
	Monitor_State.MonitorHSyncCount = 0;
	Monitor_State.MonitorVSyncCount = 0;
	Monitor_State.MonitorHorizontalCount = 0;
	Monitor_State.MonitorScanLineCount = 0;
	Monitor_State.MonitorSyncInputs = 0;
	Monitor_State.bSeenHorzInputSync = FALSE;
	Monitor_State.HorzSyncAfterCount = 0;
	Monitor_State.HorzStartAdjustment = 0;
	Monitor_State.bSeenVertInputSync = FALSE;
	Monitor_State.VertSyncAfterCount = 0;
	Monitor_State.VertStartAdjustment = 0;
	Monitor_State.bDrawSync = FALSE;
  Monitor_State.bVerticalLineCounterActive = TRUE;

#ifdef LESS_MULTS
	Render_FirstLine();
#endif

}

BOOL Monitor_DrawSync()
{
	return Monitor_State.bDrawSync;
}

void Monitor_Init()
{
	DontRender = FALSE;
}

#if 0
void	CRTC_SetRenderState(BOOL State)
{
	DontRender = State;
}

#define MONITOR_VSYNC_COUNT	(24+1)
#endif

#if 0
static int AccumulatedCycles;
static int RenderStageCount;
static CRTC_RENDER_STAGE	*pCurrentRenderStage;
static CRTC_RENDER_STAGE	RenderStages[64];


static void CRTC_EndRenderStage(void)
{
	/* set number of cycles in this stage */
	pCurrentRenderStage->Count = AccumulatedCycles;
	AccumulatedCycles = 0;
}

static void	CRTC_NextRenderStage(unsigned long Flags)
{
	CRTC_EndRenderStage();

	/* next render stage */
	pCurrentRenderStage++;
	/* update number of render stages */
	RenderStageCount++;

	/* store current MA */
	pCurrentRenderStage->MA = MALine + HCount;
	/* store rendering method to use */
	pCurrentRenderStage->RenderType = CRTC_RenderType;
	pCurrentRenderStage->Flags = Flags;
}

static void	CRTC_FirstRenderStage(void)
{
	RenderStageCount=1;
	pCurrentRenderStage = &RenderStages[0];
	pCurrentRenderStage->MA = MALine + HCount;
	pCurrentRenderStage->RenderType = CRTC_RenderType;
	pCurrentRenderStage->Flags = 0;
}
#endif

void Monitor_DoHsyncStart()
{
	Monitor_State.MonitorFlags |= MONITOR_DRAW_HSYNC;
	Monitor_UpdateGraphicsFunc();


	Monitor_State.bSeenHorzInputSync = TRUE;
	Monitor_State.HorzSyncAfterCount=0;
	Monitor_State.MonitorSyncInputs |= (1<<0);
}


#if 0
/*

  Bit 15: MA13
  Bit 14: MA12
  Bit 13: RA3
  Bit 12: RA2
  Bit 11: RA0
  Bit 10: MA9
  Bit 9: MA8
  Bit 8: MA7
  Bit 7: MA6
  Bit 6: MA5
  Bit 5: MA4
  Bit 4: MA3
  Bit 3: MA2
  Bit 2: MA1
  Bit 1: MA0
  Bit 0: CCLK
*/
#endif

#if 0
/*static void CRTC_RenderCycles(); */

/* depending on HDisp, VDisp, Reg8, and Syncs, set the rendering function */
static void     CRTC_SetRenderingFunction(void)
{
	if (Monitor_State.MonitorSyncInputs!=0)
	{
		CRTC_RenderType = CRTC_RENDER_SYNC;
	}
	else
		if (!Monitor_State.bDispEnable)
		{
			CRTC_RenderType = CRTC_RENDER_BORDER;
		}
		else
		{
			CRTC_RenderType = CRTC_RENDER_GRAPHICS;
		}
}
#endif


/* R4 range: 35-41 */
/* 34 flickers a lot */
/* 42 scrolls down */

/* 42-34 is a range of 64 lines */


/* R0 range: 62-65 */
/* screen bends and lines are not diagonal */
/* 65-62 is a range of 3 chars */

/* a normal screen is 312 lines tall, 38 is normal 4 and 3 */
/* R0 - 63, R2 = 46, 14 hsync width */
/* 18 chars of hsync, with 14 to define it */
/* 8 vsync width, R4 = 38, R7 = 30 */
/* So 8 lines of vsync, with a width of 8 */

/* 39 is 8 extra lines */
/* screen shifts down to try and match */

/* R7 = 30, R4 = 38 */
/* R7 = 30, R4 = 39  (288 to 336 lines, 312 lines is normal) 49 lines leway */




void    BrightnessControl_AdjustBrightness(RGB_CHAR *pSourceRGB, RGB_CHAR *pDestRGB, int Brightness)
{
	RGB_FLOAT SourceRGBFloat;

	float RGB_CHAR_TO_FLOAT_FACTOR = 1/255.0f;
	float RGB_FLOAT_TO_CHAR_FACTOR = 255.0f;
	/*    float RGB_FLOAT_TO_CHAR_FACTOR = 255.0f;
	   float BrightnessFloat = (float)Brightness/256.0f;
*/
	/* convert char RGB to float RGB */
	SourceRGBFloat.R = (float)(pSourceRGB->R)*RGB_CHAR_TO_FLOAT_FACTOR;
	SourceRGBFloat.G = (float)(pSourceRGB->G)*RGB_CHAR_TO_FLOAT_FACTOR;
	SourceRGBFloat.B = (float)(pSourceRGB->B)*RGB_CHAR_TO_FLOAT_FACTOR;

#ifdef POW
	SourceRGBFloat.R = pow(SourceRGBFloat.R,1/BrightnessFloat);
	SourceRGBFloat.G = pow(SourceRGBFloat.G,1/BrightnessFloat);
	SourceRGBFloat.B = pow(SourceRGBFloat.B, 1/BrightnessFloat);
#else
	SourceRGBFloat.R = SourceRGBFloat.R;
	SourceRGBFloat.G = SourceRGBFloat.G;
	SourceRGBFloat.B = SourceRGBFloat.B;
#endif
	pDestRGB->R = (char)(SourceRGBFloat.R * RGB_FLOAT_TO_CHAR_FACTOR);
	pDestRGB->G = (char)(SourceRGBFloat.G * RGB_FLOAT_TO_CHAR_FACTOR);
	pDestRGB->B = (char)(SourceRGBFloat.B * RGB_FLOAT_TO_CHAR_FACTOR);
}

int Monitor_GetHorizontalRetraceCount()
{
	return Monitor_State.MonitorHSyncCount;
}

int Monitor_GetVerticalRetraceCount()
{
	return Monitor_State.MonitorVSyncCount;
}

BOOL Monitor_SeeHsync()
{
	return Monitor_State.bSeenHorzInputSync;
}

int Monitor_CountAfterHsync()
{
	return Monitor_State.HorzSyncAfterCount;
}


BOOL Monitor_SeeVsync()
{
	return Monitor_State.bSeenHorzInputSync;
}

int Monitor_CountAfterVsync()
{
	return Monitor_State.VertSyncAfterCount;
}


int Monitor_VertStartAdjustment()
{
	return Monitor_State.VertStartAdjustment;
}

int Monitor_HorzStartAdjustment()
{
	return Monitor_State.HorzStartAdjustment;
}



void	Monitor_Cycle(int Cycles)
{
#ifdef SIMPLE_EMULATION
	// increment
	Monitor_State.MonitorHorizontalCount = (unsigned char)((CRTC_InternalState.Monitor_State.MonitorHorizontalCount+1) & 0x03f);

	Monitor_State.CharsAfterHsync++;


	if (Monitor_State.CharsAfterHsync>=64)
	{
		// end of hsync seen?
		if (HorizontalSyncWidth==0)
		{
			CRTC_InternalState.Monitor_State.CharsAfterHsync=0;


			// next line
			/* for monitor sync */
			Monitor_State.MonitorHorizontalCount = 0;

			Monitor_State.MonitorScanLineCount++;

//#ifdef LESS_MULTS
//			Render_NextLine();
//#endif

			if (CRTC_InternalState.Monitor_State.MonitorScanLineCount>=(39*8))
			{
				// new frame
				CRTC_InternalState.Monitor_State.MonitorScanLineCount = 0;
			}

		}
	}



	// new frame
	CRTC_InternalState.Monitor_State.MonitorScanLineCount = 0;

#ifdef LESS_MULTS
	Render_FirstLine();
#endif

#ifndef SIMPLE_MONITOR_EMULATION
	/* set vsync retrace */
	CRTC_InternalState.Monitor_State.VSyncRetraceCount = 312-Y_CRTC_LINE_HEIGHT;




	//next line
	/* for monitor sync */
	CRTC_InternalState.Monitor_State.MonitorHorizontalCount = 0;

	CRTC_InternalState.Monitor_State.MonitorScanLineCount++;

#ifdef LESS_MULTS
	Render_NextLine();
#endif

	if (CRTC_InternalState.Monitor_State.MonitorScanLineCount>=(39*8))
	{
		CRTC_Monitor_NewFrame();
	}





#else





	/* increment horizontal count */
	Monitor_State.MonitorHorizontalCount++;
	//Monitor_State.MonitorHorizontalCount&=0x03f;

	if (Monitor_State.MonitorHorizontalCount==64)
	{
		
		/*Monitor_State.MonitorHorizontalCount = 0; */

    /* this is activated when we see a vsync from the crtc and remains active until x number of lines */
		if (Monitor_State.MonitorFlags & MONITOR_DRAW_VSYNC)
		{
      
			Monitor_State.MonitorVSyncCount++;
			/*//printf("Vsync: %d\n", Monitor_State.MonitorVSyncCount); */
      if (Monitor_State.MonitorVSyncCount==MONITOR_VSYNC_COUNT)
			{

				Monitor_State.MonitorFlags &=~MONITOR_DRAW_VSYNC;
				Monitor_UpdateGraphicsFunc();

				Monitor_State.MonitorVSyncCount = 0;
       
        if (!Monitor_State.bVerticalLineCounterActive)
        {
          Monitor_State.bVerticalLineCounterActive = TRUE;
        }
			}
		}

		Monitor_State.MonitorHorizontalCount = 0;

    
    /* Works like this:
    
    - Monitor counts the vertical lines
    - If the number reaches a max value, it will automatically reset the counter and retrace 
    
    - if it sees vsync active between a specific range, it will then perform a retrace
    
    */
    
    if (Monitor_State.bVerticalLineCounterActive)
    {
      
      Monitor_State.MonitorScanLineCount++;
	//    if (Monitor_State.MonitorScanLineCount>=312)
	//    {
	//	    Monitor_State.MonitorScanLineCount = 0;
	//    }
      /*//printf("Line: %d\n", Monitor_State.MonitorScanLineCount);    */
        /* when we are in this range... */
        if ((Monitor_State.MonitorScanLineCount>MinLineVert) && (Monitor_State.MonitorScanLineCount<MaxLineVert))
        {    
            /*//printf("checking vsync active...\n"); */
              if ((Monitor_State.PrevMonitorSyncInputs^Monitor_State.MonitorSyncInputs) & (1<<1))
              {
                  if (Monitor_State.MonitorSyncInputs & (1<<1))
                  {
                        
                        
                  /*//printf("Reset monitor line\n");*/
                    Monitor_State.bVerticalLineCounterActive = FALSE;
                    
                    #ifdef LESS_MULTS
                          Render_FirstLine();
                    #endif
                          Monitor_State.MonitorScanLineCount = 0;
                  }              
            }
          }
          else
          {
                
              
          #ifdef LESS_MULTS
              Render_NextLine();
          #endif
                
              if (Monitor_State.MonitorScanLineCount>=310)
              {
   /*//printf("Reset monitor line overflow\n"); */
            Monitor_State.bVerticalLineCounterActive = FALSE;
          #ifdef LESS_MULTS
                Render_FirstLine();
          #endif
                Monitor_State.MonitorScanLineCount = 0;
              }
        }
    }
  }
	return;


	if (Monitor_State.bSeenHorzInputSync)
	{
		Monitor_State.HorzSyncAfterCount++;
	}

	Monitor_State.bDrawSync = FALSE;
	if (((Monitor_State.MonitorFlags & MONITOR_IN_HSYNC)!=0) || ((Monitor_State.MonitorFlags & MONITOR_IN_VSYNC)!=0))
	{
		Monitor_State.bDrawSync = TRUE;

	}


	{
		/* processing hsync? */
		if (Monitor_State.MonitorFlags & MONITOR_IN_HSYNC)
		{
			/* count hsync */
			Monitor_State.MonitorHSyncCount++;

			/*count reached? */
			if (Monitor_State.MonitorHSyncCount>=18)
			{
				/* no longer in hsync */
				Monitor_State.MonitorFlags &=~MONITOR_IN_HSYNC;


				/*we saw vertical input sync */
				if (Monitor_State.bSeenVertInputSync)
				{
					Monitor_State.VertSyncAfterCount++;
				}

				/* in vsync? */
				if (Monitor_State.MonitorFlags & MONITOR_IN_VSYNC)
				{
					/* update vsync */
					Monitor_State.MonitorVSyncCount++;

					/* is input vsync still active? */
					if (Monitor_State.MonitorSyncInputs & (1<<1))
					{
						Monitor_State.bSeenVertInputSync = FALSE;
						Monitor_State.VertStartAdjustment = 0;
					}

					if (Monitor_State.MonitorVSyncCount>=32)
					{
						Monitor_State.MonitorFlags &=~MONITOR_IN_VSYNC;

						if (Monitor_State.bSeenVertInputSync)
						{

							/* it has ended before we saw it */
							if (Monitor_State.VertSyncAfterCount<32)
							{
								/* it ended before we got to it, but it's close
								so try and get closer to it */
								Monitor_State.VertStartAdjustment = 1;
							}
							else
							{
								Monitor_State.VertStartAdjustment = -1;
							}
							Monitor_State.VertStartAdjustment = 0;
						}


						Monitor_State.MonitorScanLineCount = Monitor_State.VertStartAdjustment;

#ifdef LESS_MULTS
						Render_FirstLine();
#endif
					}
				}
				else
				{

					Monitor_State.MonitorScanLineCount++;

#ifdef LESS_MULTS
					Render_NextLine();
#endif

					if (Monitor_State.MonitorScanLineCount>=280)
					{
						Monitor_State.MonitorFlags |= MONITOR_IN_VSYNC;
						Monitor_State.MonitorVSyncCount = 0;


#if 0
						if (Monitor_State.bSeenVertInputSync)
						{
							Monitor_State.bSeenVertInputSync = FALSE;
							/* if we saw input sync before monitor sync
							 then the count indicates for how long
							*/
							if (Monitor_State.VertSyncAfterCount<=1)
							{
								Monitor_State.VertStartAdjustment = 0;
							}
							else
							{
								Monitor_State.VertStartAdjustment--;
								if (Monitor_State.VertStartAdjustment>8)
								{
									Monitor_State.VertStartAdjustment=8;
								}
							}
							Monitor_State.MonitorScanLineCount = Monitor_State.VertStartAdjustment;
						}
						else
						{
							Monitor_State.MonitorHorizontalCount -=8;
						}
#endif

					}
				}
			}
		}
		else
		{
			/* increment horizontal count */
			Monitor_State.MonitorHorizontalCount++;

			if (Monitor_State.MonitorHorizontalCount>=46)
			{
				/* we started monitor sync */
				Monitor_State.MonitorFlags |= MONITOR_IN_HSYNC;
				Monitor_State.MonitorHSyncCount = 0;
				/*  Monitor_State.MonitorHorizontalCount = ; */


				if (Monitor_State.bSeenVertInputSync)
				{
					Monitor_State.VertSyncAfterCount++;
				}

				if (Monitor_State.bSeenHorzInputSync)
				{
					Monitor_State.bSeenHorzInputSync = FALSE;
					/* if we saw input sync before monitor sync
					 then the count indicates for how long */

					if (Monitor_State.HorzSyncAfterCount<=1)
					{
						Monitor_State.HorzStartAdjustment = 0;   /*-8;*/
					}
					else
					{
						Monitor_State.HorzStartAdjustment--;
						if (Monitor_State.HorzStartAdjustment>8)
						{
							Monitor_State.HorzStartAdjustment=8;
						}
					}
					Monitor_State.HorzStartAdjustment=0;
					Monitor_State.MonitorHorizontalCount=Monitor_State.HorzStartAdjustment;
				}
				else
				{
					Monitor_State.MonitorHorizontalCount = 0;    /*-12;*/
				}

			}
		}
	}



#if 0


	/* The "InMonitorVsync" is used to render black where
	the Monitor Vsync is active. In some demos such as the
	TwinBlast demo this is very obvious. By reproducing this
	effect we can see this demo screen reproduced exactly as seen */
	if (Monitor_State.MonitorFlags & MONITOR_IN_VSYNC)
	{
		Monitor_State.MonitorVsyncCount--;

		if (Monitor_State.MonitorVsyncCount==0)
		{
			Monitor_State.MonitorFlags &= ~MONITOR_IN_VSYNC;

			CRTC_SetRenderingFunction();
		}
	}


	/* in vsync? */
	if (Monitor_State.MonitorFlags & MONITOR_VSYNC_SYNC)
	{
		Monitor_State.MonitorVSyncCount--;
		if (Monitor_State.MonitorVSyncCount<=0)
		{
			Monitor_State.MonitorFlags &=~MONITOR_VSYNC_SYNC;
		}
	}
	else
	{
		/* start a new line */
		Monitor_State.MonitorHorizontalCount = 0;
#endif

#endif
	}

#if 0
	static int CRTC_RenderMode;

	void    CRTC_SetRenderFunction(int RenderMode)
	{
		CRTC_RenderMode = RenderMode;

		switch (RenderMode)
		{
			case RENDER_MODE_STANDARD:
			{
				pCRTC_RenderGraphicsFunction = pRender_GetGraphicsDataCPC;
			}
			break;
			case RENDER_MODE_ASIC_FEATURES:
			{
				pCRTC_RenderGraphicsFunction = pRender_GetGraphicsDataPLUS;
			}
			break;
		}
	}
#endif

	void Monitor_DoHsyncEnd()
	{
		Monitor_State.MonitorFlags &= ~MONITOR_DRAW_HSYNC;
		Monitor_UpdateGraphicsFunc();

		Monitor_State.MonitorSyncInputs &= ~(1<<0);

	}

	BOOL Monitor_GetVsyncState()
	{
		return ((Monitor_State.MonitorSyncInputs & (1<<1))!=0);
	}

	BOOL Monitor_GetHsyncState()
	{
		return ((Monitor_State.MonitorSyncInputs & (1<<0))!=0);
	}

	BOOL Monitor_GetHBlank()
	{
		return ((Monitor_State.MonitorFlags & MONITOR_IN_HSYNC)!=0);
	}

	BOOL Monitor_GetVBlank()
	{
		return ((Monitor_State.MonitorFlags & MONITOR_IN_VSYNC)!=0);
	}

	int Monitor_GetHorizontalPosition()
	{
		return Monitor_State.MonitorHorizontalCount;
	}

	int Monitor_GetVerticalPosition()
	{
		return Monitor_State.MonitorScanLineCount;
	}


	void Monitor_DoVsyncStart()
	{
  /*//printf("seen vsync begin\n"); */
		Monitor_State.MonitorFlags |= MONITOR_DRAW_VSYNC;
		Monitor_UpdateGraphicsFunc();
  /* used for drawing */
		Monitor_State.MonitorVSyncCount = 0;

		Monitor_State.bSeenVertInputSync = TRUE;
		Monitor_State.VertSyncAfterCount=0;
		Monitor_State.PrevMonitorSyncInputs = Monitor_State.MonitorSyncInputs;
		Monitor_State.MonitorSyncInputs |= (1<<1);
	}

	void Monitor_DoVsyncEnd()
	{
     /*//printf("seen vsync end\n"); */

    		Monitor_State.PrevMonitorSyncInputs = Monitor_State.MonitorSyncInputs;
		Monitor_State.MonitorSyncInputs &= ~(1<<1);

/*	if (Monitor_State.MonitorFlags & MONITOR_IN_VSYNC)
	{



	}
*/
	}



#if 0
	void	Render_SetTrueColourRender(BOOL State)
	{
#if 0
		if (State==TRUE)
		{
			pRender_GetGraphicsDataCPC = Render_GetGraphicsDataAleste_TrueColour;
			pRender_GetGraphicsDataPLUS = Render_GetGraphicsDataPLUS_TrueColour;
			pRender_RenderBorder = Render_RenderBorder_Aleste_TrueColour;	/*Render_RenderBorder_TrueColour;*/
			pCRTC_RenderSync = CRTC_RenderSync_TrueColour;
		}
		else
		{
			pRender_GetGraphicsDataCPC = Render_GetGraphicsDataCPC_Paletted;
			pRender_GetGraphicsDataPLUS = Render_GetGraphicsDataPLUS_Paletted;
			pRender_RenderBorder = Render_RenderBorder_Paletted;
			pCRTC_RenderSync = CRTC_RenderSync_Paletted;
		}
#endif
		CRTC_SetRenderFunction(CRTC_RenderMode);
	}



	/* only does vertical soft scroll at this time */
	void    Render_GetGraphicsDataPLUS_Paletted(void)
	{
		/* TODO */
#if 0
		unsigned int Addr;
		unsigned int LocalMA;
		unsigned int Mask;

		/*    LocalMA = (unsigned int)(ASICCRTC_MALine + HCount); */
		LocalMA = ((CRTC_InternalState.MALine + CRTC_InternalState.HCount)<<1);

		/* get screen scrolling */
		Addr = (unsigned int)(((LocalMA & 0x03000)<<2) | ((LocalMA & 0x03ff)<<1));

		/* take address, and put in vertical line count in place of these 3 bits. */
		Addr |= ASIC_RasterMA;

		/* shift up previous 16 bits of data */
		GraphicsLong = GraphicsLong<<16;

		/* add in new 16 bits of data */
		GraphicsLong = GraphicsLong |
					   ((Z80MemoryBase[(unsigned int)(Addr)])<<8)
					   | (Z80MemoryBase[(unsigned int)(Addr+1)]);

		/* Line, Column, ActualX, ActualY */
		Mask = ASIC_BuildDisplayReturnMaskWithPixels(ASICCRTC_Line /*VisibleRasterCount*/,HCount, /*MonitorHorizontalCount, MonitorScanLineCount,*/Pixels);

		Render_Paletted_PutDataWordPLUS(Monitor_State.MonitorHorizontalCount, GraphicsLong, Monitor_State.MonitorScanLineCount, Mask,Pixels);
#endif
	}

	void    Render_GetGraphicsDataPLUS_TrueColour(void)
	{
#if 0
		unsigned int Addr;
		unsigned int LocalMA;
		unsigned int Mask;

		/*   LocalMA = (unsigned int)(ASICCRTC_MALine + HCount); */
		LocalMA = ((CRTC_InternalState.MALine + CRTC_InternalState.HCount)<<1);

		/* get screen scrolling */
		Addr = (unsigned int)(((LocalMA & 0x03000)<<2) | ((LocalMA & 0x03ff)<<1));

		/* take address, and put in vertical line count in place of these 3 bits. */
		Addr |= ASIC_RasterMA;

		/* shift up previous 16 bits of data */
		GraphicsLong = GraphicsLong<<16;

		/* add in new 16 bits of data */
		GraphicsLong = GraphicsLong |
					   ((Z80MemoryBase[(unsigned int)(Addr)])<<8)
					   | (Z80MemoryBase[(unsigned int)(Addr+1)]);

		/* Line, Column, ActualX, ActualY */
		Mask = ASIC_BuildDisplayReturnMaskWithPixels(ASICCRTC_Line /*VisibleRasterCount*/,HCount, /*MonitorHorizontalCount, MonitorScanLineCount,*/Pixels);

		/*		if (Mask==0x0ffffffff)
				{
					Render_TrueColourRGB_PutDataWord(Monitor_State.MonitorHorizontalCount, GraphicsLong, Monitor_State.MonitorScanLineCount);
				}
				else
				{
		*/
		Render_TrueColourRGB_PutDataWordPLUS(Monitor_State.MonitorHorizontalCount, GraphicsLong, Monitor_State.MonitorScanLineCount, Mask,Pixels);
		/*		} */
#endif
	}

#endif




	void	Graphics_Update(void)
	{
		Monitor_Cycle(0);

	/* frame-skip render override */
		/*if (DontRender)
			return;

		  if ((Monitor_State.MonitorFlags & (MONITOR_IN_HSYNC|MONITOR_IN_VSYNC))!=0)
		    return;
*/
		/* in vertical retrace? don't render */
/*	if (Monitor_State.VSyncRetraceCount!=0)
			return;

		if (Monitor_State.HSyncRetraceCount!=0)
		return;
*/
		/* do draw */
/*	CRTC_SetRenderingFunction();

	pCRTC_RenderGraphicsFunction();
*/
		pCRTC_RenderFunction();
#if 0
		switch (CRTC_RenderType)
		{
			case CRTC_RENDER_GRAPHICS:
			{
				pCRTC_RenderGraphicsFunction();
			}
			break;

			case CRTC_RENDER_BORDER:
			{
				pRender_RenderBorder();
			}
			break;

			case CRTC_RENDER_SYNC:
			{
				pCRTC_RenderSync();
			}
			break;
		}
#endif
	}


#endif
#endif

